require 'test/unit'
require 'tempfile'
require 'symbiosis/config_file'

class TestConfigFile < Test::Unit::TestCase

  include Symbiosis

  def setup
    @template = Tempfile.new("tc_config_file-template-")
    @template.puts("# Testing 1.. 2.. 3..\n# #{__FILE__}\n<%= $test_config_file_var %>\n% if true\nFOO\n% else\nBAR\n% end\n")
    @template.flush

    @config = Tempfile.new("tc_config_file-config-")

    $test_config_file_var = "Original value"
  end

  def teardown
  end

  def test_domain
  end

  def test_template
  end

  def test_generate_config
    config =  "# Testing 1.. 2.. 3..\n# #{__FILE__}\n#{$test_config_file_var}\nFOO\n"
    config += "# Checksum MD5 #{Digest::MD5.new.hexdigest(config)}\n"

    x = ConfigFile.new(@config.path)
    assert_equal(config, x.generate_config(@template.path))
  end

  def test_write
    config = "testing 1.. 2.. 3.."

    #
    # ConfigFile#write should write what we tell it to.
    #
    x = ConfigFile.new(@config.path)
    assert_nothing_raised{ x.write(config) }
    assert_equal(config, File.read(@config.path))
  end

  #
  # The ok? method should be overwritten by child classes to actually test the
  # configuration output.
  #
  def test_ok?
    x = ConfigFile.new(@config.path)
    assert(x.ok?) 
  end

  def test_outdated?
    x = ConfigFile.new(@config.path)
    assert_nothing_raised{ x.template = @template.path }

    #
    # The config hasn't been written yet.  Assume it is out of date.
    #
    assert_equal(true, x.outdated?)
    
    #
    # Write the configuration.
    #
    assert_nothing_raised { x.write }

    #
    # It should not be out of date now.
    #
    assert_equal(false, x.outdated?)
  
    #
    # Change our global variable.
    #
    $test_config_file_var = "New value"

    #
    # It should be out of date now.
    #
    assert_equal(true, x.outdated?)
  end

  def test_changed?
    x = ConfigFile.new(@config.path)
    assert_nothing_raised{ x.template = @template.path }

    #
    # The config is an empty file.  Assume it has been changed.
    #
    assert_equal(true, x.changed?)
    
    #
    # Generate a configuration using our template.  Once written, changed?
    # should return false.
    #
    original_config = x.generate_config
    assert_nothing_raised { x.write(original_config) }
    assert_equal(false, x.changed?)
  
    #
    # Change our config.  changed? should now return true. 
    #
    new_config =  "New Directive.\n" + original_config
    assert_nothing_raised { x.write(new_config) }
    assert_equal(true, x.changed?)

    #
    # Now write a new config with ye olde BIG WARNING.  A file with this in it
    # is *always* unchanged.
    #
    ye_olde_config = "# DO NOT EDIT THIS FILE - CHANGES WILL BE OVERWRITTEN\n Stuff goes in here.\n Blah blah blah.\n"
    assert_nothing_raised { x.write(ye_olde_config) }
    assert_equal(false, x.changed?)

    #
    # Finally write a config with no checksum or big warning.  This should always return as being changed.
    #
    custom_config = "# Hand-crafted by experts.\nBlah blah blah.\n"
    assert_nothing_raised { x.write(custom_config) }
    assert_equal(true, x.changed?)
  end

  def test_is_managed?
    x = ConfigFile.new(@config.path)
    assert_nothing_raised{ x.template = @template.path }

    #
    # The config is an empty file.  It is not managed 
    #
    assert_equal(false, x.managed?)

    #
    # Generate a configuration using our template.  Once written, managed?
    # should return false.
    #
    original_config = x.generate_config
    assert_nothing_raised { x.write(original_config) }
    assert_equal(true, x.managed?)

    #
    # Change our config.  managed? should still be true. 
    #
    new_config =  "New Directive.\n" + original_config
    assert_nothing_raised { x.write(new_config) }
    assert_equal(true, x.changed?)

    #
    # Now write a new config with ye olde BIG WARNING.  A file with this in it
    # is *always* managed.
    #
    ye_olde_config = "# DO NOT EDIT THIS FILE - CHANGES WILL BE OVERWRITTEN\n Stuff goes in here.\n Blah blah blah.\n"
    assert_nothing_raised { x.write(ye_olde_config) }
    assert_equal(true, x.managed?)

    #
    # Finally write a config with no checksum or big warning.  This is never managed
    #
    custom_config = "# Hand-crafted by experts.\nBlah blah blah.\n"
    assert_nothing_raised { x.write(custom_config) }
    assert_equal(false, x.managed?)

  end

end
