# encoding: UTF-8

require 'net/ldap'

module Mauve
  #
  # Base class for authentication.
  #
  class Authentication
    class << self
      def pre_auth_checks(login, password)
        return "Login must be a string, not a #{login.class}" if String != login.class
        return "Password must be a string, not a #{password.class}" if String != password.class
        'Login or/and password is/are empty.' if login.empty? || password.empty?
      end

      # @return [Log4r::Logger]
      def logger
        @logger ||= Log4r::Logger.new(self.to_s)
      end

      # Builds the config to open the ldap connection
      def ldap_conf(login, password)
        ldap_uri = Configuration.current.ldap_auth_url
        dn = "uid=#{login},ou=Users," + ldap_uri.dn
        conf = {
          host: ldap_uri.host,
          port: ldap_uri.port,
          base: ldap_uri.dn,
          auth: {
            method: :simple,
            username: dn,
            password: password
          }
        }
        conf[:encryption] = :simple_tls if ldap_uri.scheme == 'ldaps'
        conf
      end

      # @param [String] login
      # @param [String] password
      #
      # @return [Boolean] Success or failure.
      #
      def authenticate(login, password)
        invalid_message = pre_auth_checks(login, password)
        return login_failed(login, invalid_message) if invalid_message

        # Bind will be called automatically before the first operation (search)
        ldap = Net::LDAP.new(ldap_conf(login, password))

        # find the user as a member of the staff group
        search_result = group_search(ldap, login)
        # get the response status to check if the request was sent successfully
        # if the search finds nothing, it will return success and an empty array
        response = ldap.get_operation_result

        # return false if response is not 0 (success)
        return login_failed(login, response.message) unless response.code.zero?

        # return false if user is not in group
        fail_message = "#{login} is not in the staff group"
        return login_failed(login, fail_message) if !search_result ||
                                                    search_result.size != 1

        # success
        logger.info "Authentication for #{login} succeeded"
        true
      end

      # returns and array of Net::LDAP::Entry
      def group_search(ldap, login)
        # filters look for the uid within the group
        filter_user = Net::LDAP::Filter.eq('uid', login)
        filter_group = Net::LDAP::Filter.eq(
          'MemberOf',
          Configuration.current.ldap_auth_group
        )
        search_filter = filter_user & filter_group

        # This will return an array of Net::LDAP::Entry
        # This can be an empty array
        # The attributes option prevents the query returning the full entry
        # for efficiency and only retuns the 'dn' value
        ldap.search(filter: search_filter,
                    return_result: true,
                    attributes: ['dn'])
      end

      def login_failed(login, message)
        logger.warn "Authentication for #{login} failed: #{message}"
        # Rate limit
        sleep Configuration.current.failed_login_delay

        false
      end
    end
  end
end
