require 'mauve/notifiers/debug'
require 'net/https'
require 'cgi'
require 'uri'

module Mauve
  module Notifiers
    module Slack
      class Default
        attr_accessor :token
        attr_reader   :name

        def initialize(name)
          @name = name
        end

        def logger
          @logger ||= Log4r::Logger.new self.class.to_s.sub(/::Default$/, '')
        end

        attr_reader :gateway

        def gateway=(uri)
          @gateway = URI.parse(uri)
        end

        def send_alert(destination, alert, all_alerts, conditions = {})
          msg = prepare_message(destination, alert, all_alerts, conditions)

          uri = @gateway.dup
          uri.path = destination

          uri.query = ''

          http = Net::HTTP.new(uri.host, uri.port)

          if uri.port == 443
            http.use_ssl = true
            http.verify_mode = OpenSSL::SSL::VERIFY_PEER
          end

          response = http.post(uri.request_uri, msg,
                               'Content-Type' => 'application/json',
                               'Content-Length' => msg.length.to_s)

          if response.is_a?(Net::HTTPSuccess)
            #
            # Woo -- return true!
            #
            true
          else
            logger.warn("Failed with code #{response.code}: #{response.message}")
            logger.warn("Failed message: #{msg}")
            false
          end
        end

        protected

        def prepare_message(destination, alert, all_alerts, conditions = {})
          was_suppressed = conditions[:was_suppressed] || false
          will_suppress  = conditions[:will_suppress]  || false

          template_file = File.join(File.dirname(__FILE__), 'templates', 'slack.json.erb')

          txt = if File.exist?(template_file)
                  ERB.new(File.read(template_file)).result(binding).chomp
                else
                  logger.error("Could not find #{template_file} template")
                  alert.to_s
                end
        end

        def sanitize_message(msg)
          # Needed for Slack https://api.slack.com/docs/message-formatting#how_to_escape_characters
          msg.gsub!(/&/, '&amp;')
          msg.gsub!(/</, '&lt;')
          msg.gsub!(/>/, '&gt;')

          # For pretty display
          msg.gsub!(%r{&lt;br/?&gt;}, '\n') # <br> <br/>
          msg.gsub!(%r{&lt;/?pre&gt;}, '```') # <pre></pre>
          msg.gsub!(%r{&lt;/?p&gt;}, '\n') # <p></p>
          # Convert <a href="1">2</a> into <1|2>
          msg.gsub!(%r{&lt;a href=\\?"([^\\"]+)\\?"&gt;(.+)&lt;/a&gt;},
                    '<\1|\2>')

          # Needed to avoid breaking JSON
          msg.tr!('"', "'")

          msg
        end
      end
    end
  end
end
