#
#  Each of our jobs will have a top-level directory, containing
# subdirectories for each test.
#
#  We count the expected alerts-raised by each job, and abort if
# they do not match.
#

require 'yaml'
require 'minitest/autorun'
load "../sbin/bytemark-healthcheck"

KNOWN_MISSING=%w(apt_legacy_check apt_update_check bad_ipv6 bond_check drbd_checker megaraid_hardware_raid perm_check syslog tw_hardware_raid )

class TestJobs  < Minitest::Test

  def setup

    #
    #  Look for the test-directory.
    #
    @prefix = File.join(File.dirname(__FILE__), File.basename(__FILE__, ".rb"))

    #
    #  Look for the job-directory.
    #
    @jobsd = File.expand_path(File.join(File.dirname(__FILE__), "../etc/bytemark-healthcheck/jobs.d/"))
  end

  def test_jobs
    #
    #  Look for jobs.
    #
    Dir.glob( @jobsd + "/*" ) do |job_script|

      #
      # Skip save-files
      #
      next if ( job_script =~ /~$/ )

      assert( File.executable?( job_script ) , "Job script is not executable: #{job_script}" )

      #
      # Short name for the job
      #
      job = File.basename( job_script )

      #
      #  So the job is the name of the directory
      #
      dir = File.join(@prefix, job)

      if KNOWN_MISSING.include?(job)
        warn "Missing tests for #{job}"
        next
      end

      #
      #  Ensure the job exists.
      #
      assert(File.directory?(dir), "No tests found for #{job} in #{@prefix}")

      #
      #  Look at each test-attempt
      #
      Dir.glob( "#{dir}/*" ).each do |tc_prefix|
        driver = HealthcheckDriver.new(".",".")

        #
        # Short name for the test case
        #
        tc = File.basename(tc_prefix)

        #
        #  Setup a test environment.
        #
        ENV['TEST'] = "true"
        ENV['TEST_PREFIX'] = tc_prefix

        data = driver.run_job(job_script,1)

        # Make sure there are no failures in actually running the jobs inside the driver.
        assert_equal(0, driver.failures.keys.length, "The following jobs failed\n"+driver.failures.collect{|k,v| "* #{k}: #{v}"}.join("\n"))

        #
        #  Work out how many errors are expected
        #
        expected = File.open( "#{tc_prefix}/expected", "r" ).readlines[0].chomp!

        #
        # And check the received count matches.
        #
        assert_equal(expected.to_i, data.length, "#{job}, test case #{tc} expected to have #{expected} failures - output: #{data}.")

      end
    end
  end
end
